<?php
require_once '../config.php';
require_once '../middleware.php';

### API Blueprint: /user/utility
# POST /user/utility
# Description: Fetches app settings, links, and user block status.
# Headers:
#   - Device-ID: <device_id> (required)
#   - Session-Token: <session_token> (required)
# Request Body:
#   - user_id: integer (required)
#   - action: string (required, value: check_status)
# Response:
#   - 200: Success
#     {
#       "status": 200,
#       "message": "Status fetched",
#       "data": {
#         "update": {
#           "app_link": "https://example.com/app.apk",
#           "version": "1.0.1",
#           "version_code": 2,
#           "changelogs": "Bug fixes"
#         },
#         "maintenance": 0,
#         "vpn_allowed": 1,
#         "is_blocked": 0,
#         "links": {
#           "how_to": "https://example.com/how-to",
#           "telegram": "https://t.me/rewardapp",
#           "support": "https://example.com/support"
#         }
#       },
#       "new_session_token": "new_strong_session_token_64_chars"
#     }
#   - 400: Invalid user_id, Invalid action
#   - 401: Unauthorized
#   - 429: Too many requests
#   - 500: Database connection failed

header('Content-Type: application/json');

$device_id = $_SERVER['HTTP_DEVICE_ID'] ?? '';
$session_token = $_SERVER['HTTP_SESSION_TOKEN'] ?? '';

$method = $_SERVER['REQUEST_METHOD'];

if ($method !== 'POST') {
    sendResponse(405, 'Method not allowed');
}

// Get the encrypted payload
$encrypted_payload = file_get_contents('php://input');
if (empty($encrypted_payload)) {
    sendResponse(400, 'Empty request body');
}

// Decrypt the payload
$data = decryptRequestPayload($encrypted_payload);
if (!$data) {
    sendResponse(400, 'Invalid request body');
}

$action = $data['action'] ?? '';
$user_id = $data['user_id'] ?? 0;

if ($user_id <= 0) {
    sendResponse(400, 'Invalid user_id');
}

if ($action !== 'check_status') {
    sendResponse(400, 'Invalid action');
}

authenticateSession($user_id, $device_id, $session_token);
rateLimit($user_id);
$new_session_token = generateNewSessionToken($user_id, $device_id);

$conn = getDbConnection();

// Fetch app settings
$stmt = $conn->prepare("SELECT app_link, version, version_code, changelogs, maintenance_mode, vpn_allowed, how_to_link, telegram_link, support_link FROM app_settings WHERE id = 1");
$stmt->execute();
$result = $stmt->get_result();
$settings = $result->fetch_assoc();

// Fetch user block status
$stmt = $conn->prepare("SELECT is_blocked FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();

$response = [
    'update' => [
        'app_link' => $settings['app_link'],
        'version' => $settings['version'],
        'version_code' => $settings['version_code'],
        'changelogs' => $settings['changelogs']
    ],
    'maintenance' => $settings['maintenance_mode'],
    'vpn_allowed' => $settings['vpn_allowed'],
    'is_blocked' => $user['is_blocked'] ?? 0,
    'links' => [
        'how_to' => $settings['how_to_link'],
        'telegram' => $settings['telegram_link'],
        'support' => $settings['support_link']
    ]
];

sendResponse(200, 'Status fetched', $response, $new_session_token);

$stmt->close();
$conn->close();
?>